<?php
session_start();

// Garante que o ID do usuário está na sessão. Se não estiver, redireciona para o login.
// Isso é crucial para a segurança e para o filtro funcionar.
if (!isset($_SESSION['id'])) {
    header("Location: index.php"); // Altere 'index.php' para sua página de login, se for diferente
    exit();
}

$id_do_logado = $_SESSION['id'];

// Conectar ao banco ibo (studiolivecode_clientes.db)
try {
    $ibo_db = new SQLite3("./api/db/studiolivecode_clientes.db");
} catch (Exception $e) {
    die("Erro ao conectar ao banco studiolivecode_clientes.db: " . $e->getMessage());
}

// Conectar ou criar banco clientes.db na raiz
try {
    $clientes_db = new SQLite3("./clientes.db");
    $clientes_db->exec("CREATE TABLE IF NOT EXISTS clientes (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        nome TEXT NOT NULL,
        data_expiracao TEXT NOT NULL,
        valor REAL NOT NULL,
        whatsapp TEXT NOT NULL,
        chave_pix TEXT NOT NULL,
        nome_pix TEXT NOT NULL,
        banco_pix TEXT NOT NULL,
        id_user INTEGER DEFAULT 0 -- Coluna adicionada para vincular ao usuário
    )");

    // Adiciona a coluna id_user se ela não existir (para bancos já criados)
    // Isso é importante pra não perder dados se a tabela já existir sem essa coluna
    $result = $clientes_db->querySingle("PRAGMA table_info(clientes, id_user)");
    if (!$result) { // Se a coluna 'id_user' não existir
        $clientes_db->exec("ALTER TABLE clientes ADD COLUMN id_user INTEGER DEFAULT 0");
    }

} catch (Exception $e) {
    die("Erro ao conectar ao banco clientes.db: " . $e->getMessage());
}

// Cálculos para o dashboard (agora filtrados pelo ID do usuário logado)
$total_clientes = $clientes_db->querySingle("SELECT COUNT(*) FROM clientes WHERE id_user = $id_do_logado");
$total_ganhos = $clientes_db->querySingle("SELECT SUM(valor) FROM clientes WHERE data_expiracao >= date('now') AND id_user = $id_do_logado") ?: 0.0;
$clientes_vencidos = $clientes_db->querySingle("SELECT COUNT(*) FROM clientes WHERE data_expiracao < date('now') AND id_user = $id_do_logado");

// Cadastrar cliente
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['cadastrar'])) {
    $nome = isset($_POST['nome']) ? trim($_POST['nome']) : '';
    $data_expiracao = isset($_POST['data_expiracao']) ? trim($_POST['data_expiracao']) : '';
    $valor = isset($_POST['valor']) ? floatval($_POST['valor']) : 0.0;
    $whatsapp = isset($_POST['whatsapp']) ? trim($_POST['whatsapp']) : '';
    $chave_pix = isset($_POST['chave_pix']) ? trim($_POST['chave_pix']) : '';
    $nome_pix = isset($_POST['nome_pix']) ? trim($_POST['nome_pix']) : '';
    $banco_pix = isset($_POST['banco_pix']) ? trim($_POST['banco_pix']) : '';

    if (empty($nome) || empty($data_expiracao) || $valor <= 0 || empty($whatsapp) || empty($chave_pix) || empty($nome_pix) || empty($banco_pix)) {
        die("Erro: Todos os campos são obrigatórios e o valor deve ser maior que zero.");
    }

    try {
        $clientes_db->exec("BEGIN TRANSACTION");
        $query = "INSERT INTO clientes (nome, data_expiracao, valor, whatsapp, chave_pix, nome_pix, banco_pix, id_user) VALUES (" .
                 "'" . $clientes_db->escapeString($nome) . "', " .
                 "'" . $clientes_db->escapeString($data_expiracao) . "', " .
                 $valor . ", " .
                 "'" . $clientes_db->escapeString($whatsapp) . "', " .
                 "'" . $clientes_db->escapeString($chave_pix) . "', " .
                 "'" . $clientes_db->escapeString($nome_pix) . "', " .
                 "'" . $clientes_db->escapeString($banco_pix) . "', " .
                 $id_do_logado . ")"; // Salva o ID do usuário logado
        $clientes_db->exec($query);
        $clientes_db->exec("COMMIT");
    } catch (Exception $e) {
        $clientes_db->exec("ROLLBACK");
        die("Erro ao inserir cliente: " . $e->getMessage());
    }

    header("Location: gestor.php");
    exit();
}

// Editar cliente
// Note: A edição não altera o id_user, pois o cliente já está vinculado.
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['editar'])) {
    $id = isset($_POST['id']) ? intval($_POST['id']) : 0;
    $data_expiracao = isset($_POST['data_expiracao']) ? trim($_POST['data_expiracao']) : '';
    $valor = isset($_POST['valor']) ? floatval($_POST['valor']) : 0.0;
    $whatsapp = isset($_POST['whatsapp']) ? trim($_POST['whatsapp']) : '';
    $chave_pix = isset($_POST['chave_pix']) ? trim($_POST['chave_pix']) : '';
    $nome_pix = isset($_POST['nome_pix']) ? trim($_POST['nome_pix']) : '';
    $banco_pix = isset($_POST['banco_pix']) ? trim($_POST['banco_pix']) : '';

    if ($id <= 0 || empty($data_expiracao) || $valor <= 0 || empty($whatsapp) || empty($chave_pix) || empty($nome_pix) || empty($banco_pix)) {
        die("Erro: Todos os campos são obrigatórios e o valor deve ser maior que zero.");
    }

    try {
        $clientes_db->exec("BEGIN TRANSACTION");
        // Importante: Adicionamos o id_user no WHERE para garantir que só o próprio usuário pode editar seus clientes.
        $query = "UPDATE clientes SET " .
                 "data_expiracao = '" . $clientes_db->escapeString($data_expiracao) . "', " .
                 "valor = " . $valor . ", " .
                 "whatsapp = '" . $clientes_db->escapeString($whatsapp) . "', " .
                 "chave_pix = '" . $clientes_db->escapeString($chave_pix) . "', " .
                 "nome_pix = '" . $clientes_db->escapeString($nome_pix) . "', " .
                 "banco_pix = '" . $clientes_db->escapeString($banco_pix) . "' " .
                 "WHERE id = " . $id . " AND id_user = $id_do_logado"; // Filtra por ID e id_user
        $clientes_db->exec($query);
        $clientes_db->exec("COMMIT");
    } catch (Exception $e) {
        $clientes_db->exec("ROLLBACK");
        die("Erro ao atualizar cliente: " . $e->getMessage());
    }

    header("Location: gestor.php");
    exit();
}

// Excluir cliente
if (isset($_GET['excluir'])) {
    $id = intval($_GET['excluir']);
    try {
        // Importante: Adicionamos o id_user no WHERE para garantir que só o próprio usuário pode excluir seus clientes.
        $clientes_db->exec("DELETE FROM clientes WHERE id = $id AND id_user = $id_do_logado"); // Filtra por ID e id_user
    } catch (Exception $e) {
        die("Erro ao excluir cliente: " . $e->getMessage());
    }
    header("Location: gestor.php");
    exit();
}

// Função para última chave Pix (ainda pega a última global, não vinculada ao usuário)
// Se quiser que seja a última chave PIX CADASTRADA PELO USUÁRIO, teríamos que mudar.
function getUltimaChavePix($db, $id_user) {
    try {
        $stmt = $db->prepare("SELECT chave_pix, nome_pix, banco_pix FROM clientes WHERE id_user = ? ORDER BY id DESC LIMIT 1");
        $stmt->bindValue(1, $id_user, SQLITE3_INTEGER);
        $result = $stmt->execute();
        $row = $result->fetchArray(SQLITE3_ASSOC);
        return $row ?: ['chave_pix' => '', 'nome_pix' => '', 'banco_pix' => ''];
    } catch (Exception $e) {
        return ['chave_pix' => '', 'nome_pix' => '', 'banco_pix' => ''];
    }
}
// Chame a função passando o id_do_logado
$ultima_chave_pix_info = getUltimaChavePix($clientes_db, $id_do_logado);

?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gestor de Clientes</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css">
    <style>
        /* Reset básico */
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Arial', sans-serif;
            background-color: #5b5b5b;
            color: #333;
            line-height: 1.6;
            overflow-x: hidden; /* Evita rolagem horizontal */
        }

        /* Container principal */
        .container {
            width: 100%;
            max-width: 1000px;
            margin: 0 auto;
            padding: 15px;
        }

        /* Cabeçalho */
        h2, h3 {
            color: #FFFFFF; /* Mudado para branco, como estava no código anterior */
            margin-bottom: 20px;
            text-align: center;
        }

        /* Botão Voltar */
        .btn-back {
            display: inline-block;
            padding: 10px 20px;
            margin-bottom: 20px;
            background: #3498db;
            color: #fff;
            border-radius: 6px;
            text-decoration: none;
            font-size: 14px;
            transition: all 0.3s ease;
            text-align: center;
        }

        .btn-back:hover {
            background: #6c7a89;
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.2);
        }

        /* Dashboard */
        .dashboard {
            display: flex;
            flex-wrap: wrap;
            gap: 15px;
            margin-bottom: 30px;
            justify-content: center;
        }

        .dashboard-card {
            flex: 1;
            width: 100%;
            max-width: 300px;
            background: #fff;
            border-radius: 15px;
            padding: 15px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
            text-align: center;
            transition: transform 0.3s ease, box-shadow 0.3s ease;
        }

        .dashboard-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 16px rgba(0, 0, 0, 0.2);
        }

        .dashboard-card i {
            font-size: 30px;
            margin-bottom: 8px;
        }

        .dashboard-card h4 {
            font-size: 16px;
            color: #2c3e50;
            margin-bottom: 8px;
        }

        .dashboard-card .value {
            font-size: 24px;
            font-weight: bold;
        }

        .card-clientes {
            border-left: 5px solid #3498db;
        }

        .card-clientes i {
            color: #3498db;
        }

        .card-clientes .value {
            color: #3498db;
        }

        .card-ganhos {
            border-left: 5px solid #2ecc71;
        }

        .card-ganhos i {
            color: #2ecc71;
        }

        .card-ganhos .value {
            color: #2ecc71;
        }

        .card-vencidos {
            border-left: 5px solid #e74c3c;
        }

        .card-vencidos i {
            color: #e74c3c;
        }

        .card-vencidos .value {
            color: #e74c3c;
        }

        /* Input group (select + botão) */
        .input-group {
            display: flex;
            gap: 10px;
            margin-bottom: 25px;
            flex-wrap: wrap;
            justify-content: center;
            align-items: center;
        }

        .form-control {
            padding: 10px;
            border: 1px solid #d1d9e6;
            border-radius: 8px;
            font-size: 14px;
            width: 100%;
            max-width: 300px;
            transition: border-color 0.3s;
        }

        .form-control:focus {
            border-color: #3498db;
            outline: none;
            box-shadow: 0 0 5px rgba(52, 152, 219, 0.3);
        }

        /* Tabela */
        .table-container {
            width: 100%;
            overflow-x: auto;
        }

        .table {
            width: 100%;
            min-width: 800px; /* Ajustado para acomodar botões */
            border-collapse: collapse;
            background: #fff;
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
        }

        .table th, .table td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #e9ecef;
            font-size: 14px;
            white-space: nowrap;
            max-width: 150px;
            overflow: hidden;
            text-overflow: ellipsis;
        }

        .actions-column {
            max-width: none;
            overflow: visible;
            white-space: normal;
        }

        .table th {
            background: #3498db;
            color: #fff;
            font-weight: 600;
        }

        .table tr:nth-child(even) {
            background: #f8fafc;
        }

        .table tr:hover {
            background: #e6f0fa;
        }

        /* Botões */
        .btn {
            display: inline-block;
            padding: 8px 12px;
            border: none;
            border-radius: 6px;
            font-size: 12px;
            cursor: pointer;
            text-decoration: none;
            transition: all 0.3s ease;
            white-space: nowrap;
        }

        .btn-success {
            background: #2ecc71;
            color: #fff;
        }

        .btn-success:hover {
            background: #27ae60;
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.2);
        }

        .btn-primary {
            background: #3498db;
            color: #fff;
        }

        .btn-primary:hover {
            background: #2980b9;
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.2);
        }

        .btn-warning {
            background: #f1c40f;
            color: #fff;
        }

        .btn-warning:hover {
            background: #d4ac0d;
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.2);
        }

        .btn-danger {
            background: #e74c3c;
            color: #fff;
        }

        .btn-danger:hover {
            background: #c0392b;
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.2);
        }

        .btn-sm {
            padding: 6px 10px;
            font-size: 11px;
        }

        /* Modal */
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            z-index: 1000;
            overflow: auto;
        }

        .modal.show {
            display: block;
        }

        .modal-dialog {
            max-width: 90%;
            margin: 30px auto;
        }

        .modal-content {
            background: #fff;
            border-radius: 12px;
            box-shadow: 0 8px 16px rgba(0, 0, 0, 0.2);
            animation: slideIn 0.3s ease;
        }

        @keyframes slideIn {
            from {
                transform: translateY(-50px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }

        .modal-header {
            padding: 15px;
            border-bottom: 1px solid #e9ecef;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .modal-header h2 {
            margin: 0;
            font-size: 20px;
            color: #2c3e50;
        }

        .close {
            background: none;
            border: none;
            font-size: 20px;
            cursor: pointer;
            color: #7f8c8d;
            transition: color 0.3s;
        }

        .close:hover {
            color: #e74c3c;
        }

        .modal-body {
            padding: 15px;
        }

        .form-group {
            margin-bottom: 15px;
        }

        .form-group label {
            display: block;
            margin-bottom: 6px;
            font-weight: 500;
            color: #2c3e50;
            font-size: 14px;
        }

        .form-group input,
        .form-group select {
            width: 100%;
            padding: 10px;
            border: 1px solid #d1d9e6;
            border-radius: 8px;
            font-size: 14px;
            transition: border-color 0.3s;
        }

        .form-group input:focus,
        .form-group select:focus {
            border-color: #3498db;
            outline: none;
            box-shadow: 0 0 5px rgba(52, 152, 219, 0.3);
        }

        .form-group input[readonly] {
            background: #e9ecef;
            cursor: not-allowed;
        }

        /* Responsividade */
        @media (max-width: 768px) {
            .container {
                padding: 10px;
            }

            .dashboard {
                flex-direction: column;
                gap: 10px;
            }

            .dashboard-card {
                max-width: 100%;
                padding: 12px;
            }

            .dashboard-card i {
                font-size: 24px;
            }

            .dashboard-card h4 {
                font-size: 14px;
            }

            .dashboard-card .value {
                font-size: 20px;
            }

            .input-group {
                flex-direction: column;
                align-items: center;
            }

            .form-control {
                max-width: 100%;
                font-size: 13px;
            }

            .btn {
                padding: 8px 12px;
                font-size: 12px;
            }

            .btn-success {
                width: 100%;
                max-width: 300px;
                text-align: center;
            }

            .table th, .table td {
                padding: 8px;
                font-size: 12px;
                max-width: 100px;
            }

            .btn-sm {
                padding: 5px 8px;
                font-size: 10px;
            }

            .modal-dialog {
                margin: 15px auto;
                max-width: 95%;
            }

            .modal-header h2 {
                font-size: 18px;
            }

            .modal-body {
                padding: 12px;
            }

            .form-group label {
                font-size: 13px;
            }

            .form-group input,
            .form-group select {
                font-size: 13px;
                padding: 8px;
            }
        }

        @media (max-width: 480px) {
            .container {
                padding: 8px;
            }

            .btn-back {
                font-size: 12px;
                padding: 8px 16px;
            }

            .dashboard-card {
                padding: 10px;
            }

            .dashboard-card i {
                font-size: 20px;
            }

            .dashboard-card h4 {
                font-size: 13px;
            }

            .dashboard-card .value {
                font-size: 18px;
            }

            .table th, .table td {
                padding: 6px;
                font-size: 11px;
                max-width: 80px;
            }

            .btn-sm {
                padding: 3px 5px;
                font-size: 10px;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <a href="index.php" class="btn-back">Voltar</a>

        <div class="dashboard">
            <div class="dashboard-card card-clientes">
                <i class="fas fa-users"></i>
                <h4>Clientes Totais</h4>
                <div class="value"><?php echo $total_clientes; ?></div>
            </div>
            <div class="dashboard-card card-ganhos">
                <i class="fas fa-dollar-sign"></i>
                <h4>Ganhos Totais</h4>
                <div class="value">R$ <?php echo number_format($total_ganhos, 2, ',', '.'); ?></div>
            </div>
            <div class="dashboard-card card-vencidos">
                <i class="fas fa-exclamation-triangle"></i>
                <h4>Clientes Vencidos</h4>
                <div class="value"><?php echo $clientes_vencidos; ?></div>
            </div>
        </div>

        <div class="modal fade" id="cadastrarModal" tabindex="-1" role="dialog" aria-labelledby="cadastrarModalLabel" aria-hidden="true">
            <div class="modal-dialog">
                <div class="modal-content">
                    <div class="modal-header">
                        <h2>Cadastrar Cliente</h2>
                        <button type="button" class="close" data-dismiss="modal" aria-label="Close"><span aria-hidden="true">×</span></button>
                    </div>
                    <div class="modal-body">
                        <form method="POST" action="gestor.php">
                            <div class="form-group">
                                <label for="nome">Nome</label>
                                <select class="form-control" name="nome" id="nome" required>
                                    <option value="">Selecione um cliente</option>
                                    <?php
                                    // Pega os nomes dos clientes já cadastrados no gestor (pelo usuário logado)
                                    $existing_names_stmt = $clientes_db->prepare("SELECT nome FROM clientes WHERE id_user = ?");
                                    $existing_names_stmt->bindValue(1, $id_do_logado, SQLITE3_INTEGER);
                                    $existing_names_result = $existing_names_stmt->execute();
                                    $existing_names = [];
                                    while ($row = $existing_names_result->fetchArray()) {
                                        $existing_names[] = $row['nome'];
                                    }

                                    // Pega os títulos dos clientes da tabela ibo (do sistema do APK)
                                    // Filtrando apenas os que foram criados pelo usuário logado
                                    $res_ibo_stmt = $ibo_db->prepare("SELECT title FROM ibo WHERE id_user = ? AND title IS NOT NULL AND title != ''");
                                    $res_ibo_stmt->bindValue(1, $id_do_logado, SQLITE3_INTEGER);
                                    $res_ibo_result = $res_ibo_stmt->execute();

                                    while ($row_ibo = $res_ibo_result->fetchArray()) {
                                        $title = htmlspecialchars($row_ibo["title"]);
                                        // Só mostra na lista os clientes que ainda não foram cadastrados por ESTE USUÁRIO no gestor
                                        if (!in_array($title, $existing_names)) {
                                            echo "<option value='$title'>$title</option>";
                                        }
                                    }
                                    ?>
                                </select>
                            </div>
                            <div class="form-group">
                                <label for="data_expiracao">Data de Expiração</label>
                                <input type="date" class="form-control" name="data_expiracao" id="data_expiracao" value="<?php echo date('Y-m-d'); ?>" required>
                            </div>
                            <div class="form-group">
                                <label for="valor">Valor</label>
                                <input type="number" step="0.01" class="form-control" name="valor" id="valor" value="30" required>
                            </div>
                            <div class="form-group">
                                <label for="whatsapp">WhatsApp</label>
                                <input type="text" class="form-control" name="whatsapp" id="whatsapp" placeholder="11999999999" required>
                            </div>
                            <div class="form-group">
                                <label for="chave_pix">Chave Pix</label>
                                <input type="text" class="form-control" name="chave_pix" id="chave_pix" value="<?php echo htmlspecialchars($ultima_chave_pix_info['chave_pix']); ?>" required>
                            </div>
                            <div class="form-group">
                                <label for="nome_pix">Nome (Pix)</label>
                                <input type="text" class="form-control" name="nome_pix" id="nome_pix" value="<?php echo htmlspecialchars($ultima_chave_pix_info['nome_pix']); ?>" required>
                            </div>
                            <div class="form-group">
                                <label for="banco_pix">Banco (Pix)</label>
                                <input type="text" class="form-control" name="banco_pix" id="banco_pix" value="<?php echo htmlspecialchars($ultima_chave_pix_info['banco_pix']); ?>" required>
                            </div>
                            <input type="hidden" name="cadastrar" value="1">
                            <button type="submit" class="btn btn-success">Salvar</button>
                        </form>
                    </div>
                </div>
            </div>
        </div>

        <div class="modal fade" id="editarModal" tabindex="-1" role="dialog" aria-labelledby="editarModalLabel" aria-hidden="true">
            <div class="modal-dialog">
                <div class="modal-content">
                    <div class="modal-header">
                        <h2>Editar Cliente</h2>
                        <button type="button" class="close" data-dismiss="modal" aria-label="Close"><span aria-hidden="true">×</span></button>
                    </div>
                    <div class="modal-body">
                        <form method="POST" action="gestor.php">
                            <input type="hidden" name="id" id="edit_id">
                            <div class="form-group">
                                <label for="edit_nome">Nome</label>
                                <input type="text" class="form-control" name="nome" id="edit_nome" readonly>
                            </div>
                            <div class="form-group">
                                <label for="edit_data_expiracao">Data de Expiração</label>
                                <input type="date" class="form-control" name="data_expiracao" id="edit_data_expiracao" required>
                            </div>
                            <div class="form-group">
                                <label for="edit_valor">Valor</label>
                                <input type="number" step="0.01" class="form-control" name="valor" id="edit_valor" required>
                            </div>
                            <div class="form-group">
                                <label for="edit_whatsapp">WhatsApp</label>
                                <input type="text" class="form-control" name="whatsapp" id="edit_whatsapp" required>
                            </div>
                            <div class="form-group">
                                <label for="edit_chave_pix">Chave Pix</label>
                                <input type="text" class="form-control" name="chave_pix" id="edit_chave_pix" required>
                            </div>
                            <div class="form-group">
                                <label for="edit_nome_pix">Nome (Pix)</label>
                                <input type="text" class="form-control" name="nome_pix" id="edit_nome_pix" required>
                            </div>
                            <div class="form-group">
                                <label for="edit_banco_pix">Banco (Pix)</label>
                                <input type="text" class="form-control" name="banco_pix" id="edit_banco_pix" required>
                            </div>
                            <input type="hidden" name="editar" value="1">
                            <button type="submit" class="btn btn-success">Salvar</button>
                        </form>
                    </div>
                </div>
            </div>
        </div>

        <main role="main">
            <div class="row justify-text-center">

                <div class="input-group">
                    <select class="form-control" name="cliente" id="cliente">
                        <option value="">Selecione um cliente para cadastrar</option>
                        <?php
                        // Pega os nomes dos clientes já cadastrados no gestor (pelo usuário logado)
                        $existing_names_stmt = $clientes_db->prepare("SELECT nome FROM clientes WHERE id_user = ?");
                        $existing_names_stmt->bindValue(1, $id_do_logado, SQLITE3_INTEGER);
                        $existing_names_result = $existing_names_stmt->execute();
                        $existing_names = [];
                        while ($row = $existing_names_result->fetchArray()) {
                            $existing_names[] = $row['nome'];
                        }

                        // Pega os títulos dos clientes da tabela ibo (do sistema do APK)
                        // Filtrando apenas os que foram criados pelo usuário logado
                        $res_ibo_stmt = $ibo_db->prepare("SELECT title FROM ibo WHERE id_user = ? AND title IS NOT NULL AND title != ''");
                        $res_ibo_stmt->bindValue(1, $id_do_logado, SQLITE3_INTEGER);
                        $res_ibo_result = $res_ibo_stmt->execute();

                        while ($row_ibo = $res_ibo_result->fetchArray()) {
                            $title = htmlspecialchars($row_ibo["title"]);
                            // Só mostra na lista os clientes que ainda não foram cadastrados por ESTE USUÁRIO no gestor
                            if (!in_array($title, $existing_names)) {
                                echo "<option value='$title'>$title</option>";
                            }
                        }
                        ?>
                    </select>
                    <button class="btn btn-success" data-toggle="modal" data-target="#cadastrarModal">Cadastrar</button>
                </div>
            </div>

            <div class="row mt-4">

                <div class="table-container">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Nome</th>
                                <th>Data de Expiração</th>
                                <th>Valor</th>
                                <th>WhatsApp</th>
                                <th>Ações</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php
                            // Filtrando a consulta para mostrar APENAS os clientes do id_user logado
                            $res = $clientes_db->query("SELECT * FROM clientes WHERE id_user = $id_do_logado ORDER BY data_expiracao ASC");
                            while ($row = $res->fetchArray(SQLITE3_ASSOC)) {
                                $whatsapp = htmlspecialchars($row['whatsapp']);
                                $valor = number_format($row['valor'], 2, ',', '.');
                                // Converter data para DD/MM/YYYY
                                $data_vencimento = date('d/m/Y', strtotime($row['data_expiracao']));
                                // Buscar link_payment da tabela ibo (para o cliente específico e pelo id_user do admin)
                                $link_payment = '';
                                $stmt = $ibo_db->prepare("SELECT link_payment FROM ibo WHERE title = ? AND id_user = ?");
                                $stmt->bindValue(1, $row['nome'], SQLITE3_TEXT);
                                $stmt->bindValue(2, $id_do_logado, SQLITE3_INTEGER);
                                $result_link = $stmt->execute();
                                if ($link_row = $result_link->fetchArray(SQLITE3_ASSOC)) {
                                    $link_payment = htmlspecialchars($link_row['link_payment'] ?: '');
                                }
                                // Codificar apenas os campos dinâmicos
                                $nome_enc = urlencode($row['nome']);
                                $valor_enc = urlencode($valor);
                                $data_vencimento_enc = urlencode($data_vencimento);
                                $chave_pix_enc = urlencode($row['chave_pix']);
                                $nome_pix_enc = urlencode($row['nome_pix']);
                                $banco_pix_enc = urlencode($row['banco_pix']);
                                $link_payment_enc = urlencode($link_payment);
                                // Construir mensagem com %0A para cada linha
                                $msg = "🙋 Olá, $nome_enc,%0A%0A" .
                                        "Sua mensalidade valor de R$$valor_enc%0A" .
                                        "Vence em $data_vencimento_enc.%0A%0A" .
                                        "Para Renovar,%0A%0A" .
                                        "Chave Pix: $chave_pix_enc%0A%0A" .
                                        "Nome: $nome_pix_enc%0A%0A" .
                                        "Banco: $banco_pix_enc%0A%0A" .
                                        ($link_payment ? "🚀Link para Renovação Automática: $link_payment_enc" : "");
                                echo "<tr>";
                                echo "<td>" . htmlspecialchars($row['nome']) . "</td>";
                                echo "<td>" . $data_vencimento . "</td>";
                                echo "<td>R$ " . $valor . "</td>";
                                echo "<td>" . $whatsapp . "</td>";
                                echo "<td class='actions-column'>";
                                echo "<a href='https://wa.me/55" . $whatsapp . "?text=" . $msg . "' class='btn btn-primary btn-sm' target='_blank'>Cobrar</a> ";
                                echo "<a href='#' class='btn btn-warning btn-sm' data-toggle='modal' data-target='#editarModal' data-id='" . $row['id'] . "' data-nome='" . htmlspecialchars($row['nome']) . "' data-data_expiracao='" . $row['data_expiracao'] . "' data-valor='" . $row['valor'] . "' data-whatsapp='" . $whatsapp . "' data-chave_pix='" . htmlspecialchars($row['chave_pix']) . "' data-nome_pix='" . htmlspecialchars($row['nome_pix']) . "' data-banco_pix='" . htmlspecialchars($row['banco_pix']) . "'>Editar</a> ";
                                echo "<a href='gestor.php?excluir=" . $row['id'] . "' class='btn btn-danger btn-sm' onclick='return confirm(\"Confirmar exclusão?\")'>Excluir</a>";
                                echo "</td>";
                                echo "</tr>";
                            }
                            ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </main>
    </div>

    <script src="https://code.jquery.com/jquery-3.5.1.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/@popperjs/core@2.5.4/dist/umd/popper.min.js"></script>
    <script src="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/js/bootstrap.min.js"></script>

    <script>
        // Preencher o modal de cadastro com o cliente selecionado
        $("[data-target='#cadastrarModal']").click(function() {
            var cliente = $("#cliente").val();
            if (cliente !== "") {
                $("#nome").val(cliente);
            }
        });

        // Preencher o modal de edição com os dados do cliente
        $('#editarModal').on('show.bs.modal', function (event) {
            var button = $(event.relatedTarget);
            var id = button.data('id');
            var nome = button.data('nome');
            var data_expiracao = button.data('data_expiracao');
            var valor = button.data('valor');
            var whatsapp = button.data('whatsapp');
            var chave_pix = button.data('chave_pix');
            var nome_pix = button.data('nome_pix');
            var banco_pix = button.data('banco_pix');

            var modal = $(this);
            modal.find('#edit_id').val(id);
            modal.find('#edit_nome').val(nome);
            modal.find('#edit_data_expiracao').val(data_expiracao);
            modal.find('#edit_valor').val(valor);
            modal.find('#edit_whatsapp').val(whatsapp);
            modal.find('#edit_chave_pix').val(chave_pix);
            modal.find('#edit_nome_pix').val(nome_pix);
            modal.find('#edit_banco_pix').val(banco_pix);
        });
    </script>
</body>
</html>